import shlex

import lldb
import lldb.formatters
from jb_declarative_formatters.parsers.natvis import natvis_parse_file
from ._jb_lldb_utils import register_lldb_commands, make_absolute_name
from .jb_lldb_declarative_formatters import g_types_storage, reload_all


def __lldb_init_module(debugger, internal_dict):
    logger = lldb.formatters.Logger.Logger()
    logger >> '*' * 80
    logger >> 'JetBrains natvis proxy lldb module registered as ' + __name__

    commands_list = {
        make_absolute_name(__name__, '_cmd_load_natvis'): 'jb_renderers_load_natvis'
    }
    register_lldb_commands(debugger, commands_list)


def _cmd_load_natvis(debugger, command, exe_ctx, result, internal_dict):
    HELP_STR = 'Usage: jb_renderers_load_natvis [--no-reload] <natvis_file_path>...'
    cmd = shlex.split(command)
    if len(cmd) < 1:
        result.SetError(
            'At least one natvis file expected.\n{}'.format(HELP_STR))
        return

    opt_reload_after_register = True
    if cmd[0].startswith('--'):
        if cmd[0] == '--no-reload':
            opt_reload_after_register = False
            cmd = cmd[1:]
    if len(cmd) < 1:
        result.SetError(
            'At least one natvis file expected.\n{}'.format(HELP_STR))
        return

    register_natvis_file(cmd)
    if opt_reload_after_register:
        reload_all(debugger)


def register_natvis_file(filepaths):
    def type_viz_name_pp(type_viz_name):
        return "'" + str(type_viz_name) + "'"

    logger = lldb.formatters.Logger.Logger()
    for filepath in filepaths:
        logger >> "Parsing " + filepath

        for type_viz in natvis_parse_file(filepath, logger):
            logger >> "Register types: " + ', '.join(map(type_viz_name_pp, type_viz.type_viz_names))
            g_types_storage.add_type(type_viz)
