/***********************************************************************/
/*                                                                     */
/*                      ADOBE CONFIDENTIAL                             */
/*                   _ _ _ _ _ _ _ _ _ _ _ _ _                         */
/*                                                                     */
/*  Copyright 2016 Adobe Systems Incorporated                          */
/*  All Rights Reserved.                                               */
/*                                                                     */
/* NOTICE:  All information contained herein is, and remains           */
/* the property of Adobe Systems Incorporated and its suppliers,       */
/* if any.  The intellectual and technical concepts contained          */
/* herein are proprietary to Adobe Systems Incorporated and its        */
/* suppliers and are protected by all applicable intellectual property */
/* laws, including trade secret and copyright laws.                    */
/* Dissemination of this information or reproduction of this material  */
/* is strictly forbidden unless prior written permission is obtained   */
/* from Adobe Systems Incorporated.                                    */
/*                                                                     */
/***********************************************************************/

function Controller()
{
	Controller.instance = this;

	var workflowStorage = IWorkflowStorage.create();
	var currentWorkflow = null;
	var currentProcessor = null;
	var surface = new PanelSurface();
	var updateWorkflowList = true;
	var workflowsJob = null;
	var displayJob = null;
	
	// initialize
	(function()
	{
		switchSurface(PANELSURFACE_WORKFLOW_LIST);
		
		try
		{
			displayAllWorkflowTitles();
		}
		catch(exc)
		{
			exclog(exc);
		}
	})();
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Display start screen
	//
	this.displayWorkflowList = function()
	{
		cancelJobs();
		switchSurface(PANELSURFACE_WORKFLOW_LIST);

		try
		{
			displayAllWorkflowTitles();
		}
		catch(exc)
		{
			exclog(exc);
		}
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Select a workflow
	//
	this.selectWorkflow = function(/*[String]*/ inID)
	{
		cancelJobs();

		var thisObj = this;
		currentProcessor = null;
		var job = workflowStorage.getWorkflow(inID);
		job.wait(function(/*[Workflow]*/ inWorkflow)
		{
			currentWorkflow = inWorkflow;
		
			if (isValidProperty(currentWorkflow))
			{
				try
				{
					currentProcessor = new WorkflowProcessor(currentWorkflow, surface);
					switchSurface(PANELSURFACE_WORKFLOW);
					currentProcessor.initialize();
				}
				catch(exc)
				{
					exclog(exc);
					currentProcessor = null;
					// back to start surface
					thisObj.displayWorkflowList();
				}
			}
			else
			{
				currentWorkflow = null;
			}
		});
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	this.onMessage = function(/*[String]*/ inCommand)
	{
		switch(inCommand)
		{
			// Start workflow processing
			//
			case CLICKEVENT_WORKFLOW_START:
			{
				if (isValidProperty(currentProcessor))
				{
					switchSurface(PANELSURFACE_WORKFLOW_STEP);
					incWorkflowAttempts(currentWorkflow.id);
					currentProcessor.start();
				}
			}
			break;
			
			// Restart current workflow
			//
			case WorkflowProcessor.RESTART:
			{
				if (isValidProperty(currentWorkflow))
				{
					this.selectWorkflow(currentWorkflow.id);
				}
			}
			break;
			
			// Repeat the current step of a processing workflow
			//
			case CLICKEVENT_WORKFLOW_STEP_REPEAT:
			{
				if (isValidProperty(currentProcessor))
				{
					switchSurface(PANELSURFACE_WORKFLOW_STEP);
					currentProcessor.refreshStep();
				}
			}
			break;
			
			// Cancel workflow procesing
			//
			case CLICKEVENT_WORKFLOW_CANCEL:
			{
				if (isValidProperty(currentProcessor))
				{
					if (currentProcessor.isStarted())
					{
						switchSurface(PANELSURFACE_WORKFLOW_CANCELED);
						currentProcessor.cancel();
					}
					else
					{
						this.displayWorkflowList();
					}
				}
			}
			break;
			
			case CLICKEVENT_WORKFLOW_STEP_BACK:
			{
				if (isValidProperty(currentProcessor))
				{
					switchSurface(PANELSURFACE_WORKFLOW_STEP);
					currentProcessor.prevStep();
				}
			}
			break;
			
			case WorkflowProcessor.FINISHED_NEXT_STEP:
			case CLICKEVENT_WORKFLOW_STEP_NEXT:
			case CLICKEVENT_WORKFLOW_STEP_SKIP:
			{
				if (isValidProperty(currentProcessor))
				{
					switchSurface(PANELSURFACE_WORKFLOW_STEP);
					currentProcessor.nextStep();
				}
			}
			break;
			
			case WorkflowProcessor.STARTED:
			case WorkflowProcessor.SWITCH_STEP_NEXT:
			case WorkflowProcessor.SWITCH_STEP_PREV:
			{
				logAction(inCommand, currentProcessor, currentWorkflow);
			}
			break;
			
			// Execute the actual step (if expression is available)
			//
			case CLICKEVENT_WORKFLOW_STEP_EXEC:
			{
				// TODO: execute workflow step
			}
			break;
		}
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Callback function: Workflow processing finished
	//
	this.onWorkflowFinished = function(/*[String]*/ inWhy, /*[Any]*/ inData)
	{
		switch (inWhy)
		{
			case WorkflowProcessor.FINISHED_SUCCESS:
			{
				logAction(WorkflowProcessor.FINISHED_SUCCESS, currentProcessor, currentWorkflow);
				switchSurface(PANELSURFACE_WORKFLOW_SUCCEEDED);
				var display = new WorkflowDisplayAdapter(surface, currentWorkflow);
				display.showSuccess();
			}
			break;
			
			case WorkflowProcessor.FINISHED_FAILURE:
			{
				switchSurface(PANELSURFACE_WORKFLOW_STEP_FAILURE);
				var display = new WorkflowDisplayAdapter(surface, currentWorkflow);
				
				if (isValidProperty(currentProcessor))
				{
					display.showFailure(currentProcessor.getCurrentStep(), inData);
				}
			}
			break;
			
			case WorkflowProcessor.FINISHED_CANCELED:
			{
				logAction(WorkflowProcessor.FINISHED_CANCELED, currentProcessor, currentWorkflow);
			}
			break;

			case CLICKEVENT_WORKFLOW_SUCCESS_COMMIT:
			case CLICKEVENT_WORKFLOW_CANCEL_COMMIT:
			{
				currentWorkflow = null;
				currentProcessor = null;
				this.displayWorkflowList();
			}
			break;
		}
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Callback: Workflow step succeeded
	//
	this.onWorkflowStepSucceeded = function()
	{
		switchSurface(PANELSURFACE_WORKFLOW_STEP_SUCCESS);
		var display = new WorkflowDisplayAdapter(surface, currentWorkflow);

		if (isValidProperty(currentProcessor))
		{
			display.showStepSuccess(currentProcessor.getCurrentStep());
		}
	}

	// private ///////////////////////////////////////////////////////////////////

	//////////////////////////////////////////////////////////////////////////////
	//
	// Display given workflow titles
	//
	function displayWorkflowTitles(/*[Array]*/ inWorkflows)
	{
		if (surface.isCurrent(PANELSURFACE_WORKFLOW_LIST) &&
			isValidProperty(inWorkflows) &&
			isValidProperty(inWorkflows.length))
		{
			if (isValidProperty(displayJob))
			{
				displayJob.cancel();
			}

			var display = new WorkflowDisplayAdapter(surface);
			displayJob = display.showTitles(CONTENTLOCATOR_WORKFLOW_LIST, inWorkflows, onWorkflowSelect);
		}
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Display all workflow titles
	//
	function displayAllWorkflowTitles()
	{
		function sortWorkflows(/*[Workflow]*/ inWF1, /*[Workflow]*/ inWF2)
		{
			if (inWF1.date < inWF2.date)
			{
				return +1;
			}
			if (inWF1.date > inWF2.date)
			{
				return -1;
			}
			return 0;
		}

		// cancel running job
		//
		cancelJobs();

		// request workflows
		//
		workflowsJob = workflowStorage.getWorkflows();

		// process available workflows
		//
		var finished = workflowsJob.isStatus(IJob.STATUS_FINISHED | IJob.STATUS_SUCCESS) && !workflowsJob.isStatus(IJob.STATUS_MULTIPLE);

		try
		{
			var requestWorkflowsResults = workflowsJob.getResult();
			var workflows = [];

			forEach(requestWorkflowsResults, function(/*[JobResult]*/requestWorkflowsResult)
			{
				workflows = workflows.concat(requestWorkflowsResult);
			});

			workflows.sort(sortWorkflows);
			displayWorkflowTitles(workflows);
		}
		catch(exc)
		{
			exclog(exc);
		}

		// wait for remaining workflows
		//
		if (finished)
		{
			workflowsJob.cancel();
		}
		else
		{
			workflowsJob.wait(function ()
			{
				if (workflowsJob.isStatus(IJob.STATUS_FINISHED))
				{
					try
					{
						var workflows = workflowsJob.getResult();
						workflows.sort(sortWorkflows);
						displayWorkflowTitles(workflows);
					}
					catch (exc)
					{
						exclog(exc);
					}
				}
			});
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Switch panel display
	//
	function switchSurface(/*[String]*/ inName)
	{
		// workflowlist	- list of all available workflows
		// workflow		- a single in process workflow
		surface.displaySurface(inName);
		displayOffline();
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Display offline state
	//
	function displayOffline()
	{
		surface.displayByID("offline", (navigator.onLine ? "" : "Offline"));
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Cancel pending jobs
	//
	function cancelJobs()
	{
		if (isValidProperty(workflowsJob))
		{
			workflowsJob.cancel();
		}
		if (isValidProperty(displayJob))
		{
			displayJob.cancel();
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Increase number of attempts of the workflow
	//
	function incWorkflowAttempts(/*[String]*/ inWorkflowID)
	{
		var job = new ScriptJob('incWorkflowAttempts("' + inWorkflowID + '");');
		job.wait(null, function(/*[String]*/ inError)
		{
			dbglogError(inError);
		});
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Return number of attempts of the workflow
	//
	function getWorkflowAttempts(/*[String]*/ inWorkflowID, /*[Function]*/ inCallback)
	{
		var script = 'getWorkflowAttempts("' + inWorkflowID + '");';
		var job = new ScriptJob(script);
		job.wait(function(/*[Number]*/ inResult)
		{
			inCallback(inResult);
		},
		function(/*[String]*/ inError)
		{
			dbglogError(inError);
			inCallback(-1);
		});
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Log data
	//
	function logAction(/*[String]*/ inWhy, /*[WorkflowProcessor]*/ inProcessor, /*[Workflow]*/ inWorkflow)
	{
		if (isValidProperty(inProcessor))
		{
			var timeMeasure = inProcessor.getMeasurements();

			var workflowActions = [];
			var actions = [];
			var id = inWorkflow.id;
			var name = inWorkflow.name;
			var session = inWorkflow.session;
			var workflowDuration = -1;
			var stepDuration = -1;
			var stepNumber = inProcessor.getCurrentStep();
			var retry = inProcessor.getRetry();
			var failure = inProcessor.isFailure();
			var interactive = false;
			var numSteps = inWorkflow.getStepLength();
			var stepInteractivity = "no step interactivity";

			if (isValidProperty(timeMeasure['workflow']))
			{
				workflowDuration = timeMeasure['workflow'].getElapsed();
			}

			if (isValidProperty(timeMeasure['step_total' + stepNumber]))
			{
				stepDuration = timeMeasure['step_total' + stepNumber].getElapsed();
			}

			if (!isNaN(stepNumber))
			{
				var step = inWorkflow.getStep(stepNumber);
				if (isValidProperty(step) && step.getTriggers().length > 0)
				{
					stepInteractivity = "step interactivity"

					if (inProcessor.isSuccess())
					{
						stepInteractivity += " success";
					}
					else if (failure)
					{
						stepInteractivity += " failure";
					}
				}
			}

			switch (inWhy)
			{
				case WorkflowProcessor.STARTED:
				{
					actions.push('WorkflowStarted');
					workflowActions.push(true);
				}
				break;

				case WorkflowProcessor.FINISHED_SUCCESS:
				{
					actions.push('WorkflowStepSuccess');
					workflowActions.push(false);
					actions.push('WorkflowSuccess');
					workflowActions.push(true);
				}
				break;

				case WorkflowProcessor.FINISHED_CANCELED:
				{
					actions.push(failure ? 'WorkflowCanceledFailure' : 'WorkflowCanceled');
					workflowActions.push(false);
				}
				break;

				case WorkflowProcessor.SWITCH_STEP_NEXT:
				case WorkflowProcessor.SWITCH_STEP_PREV:
				{
					actions.push('WorkflowStepSuccess');
					workflowActions.push(false);
				}
				break;
			}

			getWorkflowAttempts(inWorkflow.id, function(attempts)
			{
				function createScript(/*[String]*/ inAction, /*[String]*/ inWorkflowAction)
				{
					var script = 'logAction(';
					script += inWorkflowAction;
					script += ',';
					script += '"' + inAction + '"';
					script += ',';
					script += '"' + id + '"';
					script += ',';
					script += '"' + name + '"';
					script += ',';
					script += '"' + session + '"';
					script += ',';
					script += attempts;
					script += ',';
					script += stepNumber;
					script += ',';
					script += retry;
					script += ',';
					script += '"' + stepInteractivity + '"';
					script += ',';
					script += workflowDuration;
					script += ',';
					script += stepDuration;
					script += ',';
					script += numSteps;
					script += ');';

					return script;
				}

				var esscript = '';

				for (var i=0; i<actions.length; i++)
				{
					esscript += createScript(actions[i], workflowActions[i]);
				}

				var job = new ScriptJob(esscript);
				job.wait(null, function(/*[String]*/ inError)
				{
					dbglogError(inError);
				});
			});
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Register for broadcaster messages
	//
	gBroadcaster.registerHandler(CLICKEVENT_WORKFLOW_START, this.onMessage, this);
	gBroadcaster.registerHandler(CLICKEVENT_WORKFLOW_STEP_REPEAT, this.onMessage, this);
	gBroadcaster.registerHandler(CLICKEVENT_WORKFLOW_CANCEL, this.onMessage, this);
	gBroadcaster.registerHandler(CLICKEVENT_WORKFLOW_STEP_BACK, this.onMessage, this);
	gBroadcaster.registerHandler(CLICKEVENT_WORKFLOW_STEP_NEXT, this.onMessage, this);
	gBroadcaster.registerHandler(CLICKEVENT_WORKFLOW_STEP_SKIP, this.onMessage, this);
	gBroadcaster.registerHandler(CLICKEVENT_WORKFLOW_STEP_EXEC, this.onMessage, this);

	gBroadcaster.registerHandler(WorkflowProcessor.RESTART, this.onMessage, this);
	gBroadcaster.registerHandler(WorkflowProcessor.FINISHED_NEXT_STEP, this.onMessage, this);
	gBroadcaster.registerHandler(WorkflowProcessor.STARTED, this.onMessage, this);
	gBroadcaster.registerHandler(WorkflowProcessor.SWITCH_STEP_NEXT, this.onMessage, this);
	gBroadcaster.registerHandler(WorkflowProcessor.SWITCH_STEP_PREV, this.onMessage, this);

	gBroadcaster.registerHandler(CLICKEVENT_WORKFLOW_SUCCESS_COMMIT, this.onWorkflowFinished, this);
	gBroadcaster.registerHandler(CLICKEVENT_WORKFLOW_CANCEL_COMMIT, this.onWorkflowFinished, this);

	gBroadcaster.registerHandler(WorkflowProcessor.FINISHED_SUCCESS, this.onWorkflowFinished, this);
	gBroadcaster.registerHandler(WorkflowProcessor.FINISHED_FAILURE, this.onWorkflowFinished, this);
	gBroadcaster.registerHandler(WorkflowProcessor.FINISHED_CANCELED, this.onWorkflowFinished, this);

	gBroadcaster.registerHandler(WorkflowProcessor.FINISHED_STEP_SUCCESS, this.onWorkflowStepSucceeded, this);
}

